{ *********************************************************************** }
{                                                                         }
{ Delphi Visual Component Library                                         }
{                                                                         }
{ Copyright (c) 2004 Borland Software Corporation                         }
{                                                                         }
{ *********************************************************************** }

unit Borland.Vcl.ClrXmlDom;

interface

uses
  System.Xml, System.Xml.Schema, System.IO, ActiveX, Variants, Classes, XmlDom;

const

  SCLRXML = 'CLRXML';  { Do not localize }

type

{ XmlNodeRef }

  XmlNodeRef = interface
    ['{5EF5DAA1-2729-11D4-83DA-00C04F60B2DD}']
    function GetXMLDOMNode: XmlNode;
  end;

{ TCLRDOMInterface }

  TCLRDOMInterface = class(TInterfacedObject)
  end;

{ TCLRDOMImplementation }

  TCLRDOMImplementation = class(TCLRDOMInterface, IDOMImplementation)
  private
    FCLRDOMImpl: XmlImplementation;
  protected
     { IDOMImplementation }
    function hasFeature(const feature, version: DOMString): WordBool;
    function createDocumentType(const qualifiedName, publicId,           { DOM Level 2 }
      systemId: DOMString): IDOMDocumentType;
    function createDocument(const namespaceURI, qualifiedName: DOMString;
      doctype: IDOMDocumentType): IDOMDocument;
  public
    constructor Create(DOMImpl: XmlImplementation);
    property CLRDOMImpl: XmlImplementation read FCLRDOMImpl;
  end;

{ TCLRDOMNode }

  TCLRDOMNode = class;
  TCLRDOMNodeClass = class of TCLRDOMNode;

  TCLRDOMNode = class(TCLRDOMInterface, XmlNodeRef, IDOMNode, IDOMNodeEx,
    IDOMNodeSelect)
  private
    FCLRNode: XmlNode;
    FChildNodes: IDOMNodeList;
    FAttributes: IDOMNamedNodeMap;
    FOwnerDocument: IDOMDocument;
  protected
    { XmlNodeRef }
    function GetXMLDOMNode: XmlNode;
    { IDOMNode }
    function get_nodeName: DOMString;
    function get_nodeValue: DOMString;
    procedure set_nodeValue(value: DOMString);
    function get_nodeType: DOMNodeType;
    function get_parentNode: IDOMNode;
    function get_childNodes: IDOMNodeList;
    function get_firstChild: IDOMNode;
    function get_lastChild: IDOMNode;
    function get_previousSibling: IDOMNode;
    function get_nextSibling: IDOMNode;
    function get_attributes: IDOMNamedNodeMap;
    function get_ownerDocument: IDOMDocument;
    function get_namespaceURI: DOMString;
    function get_prefix: DOMString;
    function get_localName: DOMString;
    function insertBefore(const newChild, refChild: IDOMNode): IDOMNode;
    function replaceChild(const newChild, oldChild: IDOMNode): IDOMNode;
    function removeChild(const childNode: IDOMNode): IDOMNode;
    function appendChild(const newChild: IDOMNode): IDOMNode;
    function hasChildNodes: WordBool;
    function cloneNode(deep: WordBool): IDOMNode;
    procedure normalize;
    function supports(const feature, version: DOMString): WordBool;
    { IDOMNodeEx }
    function get_text: DOMString;
    function get_xml: DOMString;
    procedure transformNode(const stylesheet: IDOMNode; var output: WideString); overload; 
    procedure transformNode(const stylesheet: IDOMNode; const output: IDOMDocument); overload; 
    { IDOMNodeSelect }
    function selectNode(const nodePath: WideString): IDOMNode;
    function selectNodes(const nodePath: WideString): IDOMNodeList;
    procedure set_text(const Value: DOMString);
  public
    constructor Create(ANode: XmlNode); virtual;
    property CLRNode: XmlNode read FCLRNode;
  end;

{ TCLRDOMNodeList }

  TCLRDOMNodeList = class(TCLRDOMInterface, IDOMNodeList)
  private
     FCLRNodeList: XmlNodeList;
  protected
    { IDOMNodeList }
    function get_item(index: Integer): IDOMNode;
    function get_length: Integer;
  public
    constructor Create(ANodeList: XmlNodeList);
    property CLRNodeList: XmlNodeList read FCLRNodeList;
  end;

{ TCLRDOMNamedNodeMap }

  TCLRDOMNamedNodeMap = class(TCLRDOMInterface, IDOMNamedNodeMap)
  private
    FCLRNamedNodeMap: XmlNamedNodeMap;
  protected
    { IDOMNamedNodeMap }
    function get_item(index: Integer): IDOMNode;
    function get_length: Integer;
    function getNamedItem(const name: DOMString): IDOMNode;
    function setNamedItem(const newItem: IDOMNode): IDOMNode;
    function removeNamedItem(const name: DOMString): IDOMNode;
    function getNamedItemNS(const namespaceURI, localName: DOMString): IDOMNode;
    function setNamedItemNS(const arg: IDOMNode): IDOMNode;
    function removeNamedItemNS(const namespaceURI, localName: DOMString): IDOMNode;
  public
    constructor Create(ANamedNodeMap: XmlNamedNodeMap);
    property CLRNamedNodeMap: XmlNamedNodeMap read FCLRNamedNodeMap;
  end;

{ TCLRDOMCharacterData }

  TCLRDOMCharacterData = class(TCLRDOMNode, IDOMCharacterData)
  private
    function GetCLRCharacterData: XmlCharacterData;
  protected
    { IDOMCharacterData }
    function get_data: DOMString;
    procedure set_data(const data: DOMString);
    function get_length: Integer;
    function substringData(offset, count: Integer): DOMString;
    procedure appendData(const data: DOMString);
    procedure insertData(offset: Integer; const data: DOMString);
    procedure deleteData(offset, count: Integer);
    procedure replaceData(offset, count: Integer; const data: DOMString);
  public
    property CLRCharacterData: XmlCharacterData read GetCLRCharacterData;
  end;

{ TCLRDOMAttr }

  TCLRDOMAttr = class(TCLRDOMNode, IDOMAttr)
  private
    function GetCLRAttribute: XmlAttribute;
  protected
    { Property Get/Set }
    function get_name: DOMString;
    function get_specified: WordBool;
    function get_value: DOMString;
    procedure set_value(const attributeValue: DOMString);
    function get_ownerElement: IDOMElement;
    { Properties }
    property name: DOMString read get_name;
    property specified: WordBool read get_specified;
    property value: DOMString read get_value write set_value;
    property ownerElement: IDOMElement read get_ownerElement;
  public
    property CLRAttribute: XmlAttribute read GetCLRAttribute;
  end;

{ TCLRDOMElement }

  TCLRDOMElement = class(TCLRDOMNode, IDOMElement)
  private
    function GetCLRElement: XmlElement;
  protected
    { IDOMElement }
    function get_tagName: DOMString;
    function getAttribute(const name: DOMString): DOMString;
    procedure setAttribute(const name, value: DOMString);
    procedure removeAttribute(const name: DOMString);
    function getAttributeNode(const name: DOMString): IDOMAttr;
    function setAttributeNode(const newAttr: IDOMAttr): IDOMAttr;
    function removeAttributeNode(const oldAttr: IDOMAttr): IDOMAttr;
    function getElementsByTagName(const name: DOMString): IDOMNodeList;
    function getAttributeNS(const namespaceURI, localName: DOMString): DOMString;
    procedure setAttributeNS(const namespaceURI, qualifiedName, value: DOMString);
    procedure removeAttributeNS(const namespaceURI, localName: DOMString);
    function getAttributeNodeNS(const namespaceURI, localName: DOMString): IDOMAttr;
    function setAttributeNodeNS(const newAttr: IDOMAttr): IDOMAttr;
    function getElementsByTagNameNS(const namespaceURI,
      localName: DOMString): IDOMNodeList;
    function hasAttribute(const name: DOMString): WordBool;
    function hasAttributeNS(const namespaceURI, localName: DOMString): WordBool; 
    procedure normalize; 
  public
    property CLRElement: XmlElement read GetCLRElement;
  end;

{ TCLRDOMText }

  TCLRDOMText = class(TCLRDOMCharacterData, IDOMText)
  protected
    function splitText(offset: Integer): IDOMText; 
  end;

{ TCLRDOMComment }

  TCLRDOMComment = class(TCLRDOMCharacterData, IDOMComment)
  end;

{ TCLRDOMCDATASection }

  TCLRDOMCDATASection = class(TCLRDOMText, IDOMCDATASection)
  end;

{ TCLRDOMDocumentType }

  TCLRDOMDocumentType = class(TCLRDOMNode, IDOMDocumentType)
  private
    FEntities: IDOMNamedNodeMap;
    FNotations: IDOMNamedNodeMap;
    function GetCLRDocumentType: XmlDocumentType;
  protected
    { IDOMDocumentType }
    function get_name: DOMString;
    function get_entities: IDOMNamedNodeMap;
    function get_notations: IDOMNamedNodeMap;
    function get_publicId: DOMString;
    function get_systemId: DOMString;
    function get_internalSubset: DOMString; 
  public
    property CLRDocumentType: XmlDocumentType read GetCLRDocumentType;
  end;

{ TCLRDOMNotation }

  TCLRDOMNotation = class(TCLRDOMNode, IDOMNotation)
  private
    function GetCLRNotation: XmlNotation;
  protected
    { IDOMNotation }
    function get_publicId: DOMString;
    function get_systemId: DOMString;
  public
    property CLRNotation: XmlNotation read GetCLRNotation;
  end;

{ TCLRDOMEntity }

  TCLRDOMEntity = class(TCLRDOMNode, IDOMEntity)
  private
    function GetCLREntity: XmlEntity;
  protected
    { IDOMEntity }
    function get_publicId: DOMString;
    function get_systemId: DOMString;
    function get_notationName: DOMString;
  public
    property CLREntity: XmlEntity read GetCLREntity;
  end;

{ TCLRDOMEntityReference }

  TCLRDOMEntityReference = class(TCLRDOMNode, IDOMEntityReference)
  end;

{ TCLRDOMProcessingInstruction }

  TCLRDOMProcessingInstruction = class(TCLRDOMNode, IDOMProcessingInstruction)
  private
    function GetCLRProcessingInstruction: XmlProcessingInstruction;
  protected
    { IDOMProcessingInstruction }
    function get_target: DOMString;
    function get_data: DOMString;
    procedure set_data(const value: DOMString);
  public
    property CLRProcessingInstruction: XmlProcessingInstruction read GetCLRProcessingInstruction;
  end;

{ TCLRDOMDocumentFragment }

  TCLRDOMDocumentFragment = class(TCLRDOMNode, IDOMDocumentFragment)
  end;

{ TCLRDOMDocument }

  TCLRDOMDocument = class(TCLRDOMNode, IDOMDocument, IDOMParseOptions, IDOMPersist,
    IDOMParseError, IDOMXMLProlog)
  private
    FValidate: Boolean;
    FResolveExternals: Boolean;
    FParseError: System.SystemException;
    function GetCLRDocument: XmlDocument;
    function InternalLoad(Reader: XmlReader): WordBool;
  protected
    { IDOMDocument }
    function get_doctype: IDOMDocumentType;
    function get_domImplementation: IDOMImplementation;
    function get_documentElement: IDOMElement;
    procedure set_documentElement(const docElement: IDOMElement);
    function createElement(const tagName: DOMString): IDOMElement;
    function createDocumentFragment: IDOMDocumentFragment;
    function createTextNode(const data: DOMString): IDOMText;
    function createComment(const data: DOMString): IDOMComment;
    function createCDATASection(const data: DOMString): IDOMCDATASection;
    function createProcessingInstruction(const target,
      data: DOMString): IDOMProcessingInstruction;
    function createAttribute(const name: DOMString): IDOMAttr;
    function createEntityReference(const name: DOMString): IDOMEntityReference;
    function getElementsByTagName(const tagName: DOMString): IDOMNodeList;
    function importNode(importedNode: IDOMNode; deep: WordBool): IDOMNode;
    function createElementNS(const namespaceURI,
      qualifiedName: DOMString): IDOMElement;
    function createAttributeNS(const namespaceURI,
      qualifiedName: DOMString): IDOMAttr;
    function getElementsByTagNameNS(const namespaceURI,
      localName: DOMString): IDOMNodeList;
    function getElementById(const elementId: DOMString): IDOMElement;
    { IDOMParseOptions }
    function get_async: Boolean;
    function get_preserveWhiteSpace: Boolean;
    function get_resolveExternals: Boolean;
    function get_validate: Boolean;
    procedure set_async(Value: Boolean);
    procedure set_preserveWhiteSpace(Value: Boolean);
    procedure set_resolveExternals(Value: Boolean);
    procedure set_validate(Value: Boolean);
    { IDOMPersist }
    function get_xml: DOMString;
    function asyncLoadState: Integer;
    function load(source: OleVariant): WordBool;
    function loadFromStream(const stream: TStream): WordBool; overload;
    function loadxml(const Value: DOMString): WordBool;
    procedure save(destination: OleVariant);
    procedure saveToStream(const stream: TStream); overload;
    procedure set_OnAsyncLoad(const Sender: TObject;
      EventHandler: TAsyncEventHandler);
    function loadFromStream(const stream: System.IO.Stream): WordBool; overload;
    procedure saveToStream(const stream: System.IO.Stream); overload;
    { IDOMParseError }
    function get_errorCode: Integer;
    function get_url: WideString;
    function get_reason: WideString;
    function get_srcText: WideString;
    function get_line: Integer;
    function get_linepos: Integer;
    function get_filepos: Integer;
    { IDOMProlog }
    function get_Encoding: DOMString;
    function get_Standalone: DOMString;
    function get_Version: DOMString;
    procedure set_Encoding(const Value: DOMString);
    procedure set_Standalone(const Value: DOMString);
    procedure set_Version(const Value: DOMString);
  public
    property CLRDocument: XmlDocument read GetCLRDocument;
  end;

{ TCLRDOMImplementationFactory }

  TCLRDOMImplementationFactory = class(TDOMVendor)
  public
    function DOMImplementation: IDOMImplementation; override;
    function Description: String; override;
  end;

var
  CLRXML_DOM: TCLRDOMImplementationFactory;

const
  NodeTypeMap: array[ELEMENT_NODE..NOTATION_NODE] of XmlNodeType =
    (XmlNodeType.Element, XmlNodeType.Attribute, XmlNodeType.Text,
      XmlNodeType.CDATA, XmlNodeType.EntityReference, XmlNodeType.Entity,
      XmlNodeType.ProcessingInstruction, XmlNodeType.Comment,
      XmlNodeType.Document, XmlNodeType.DocumentType,
      XmlNodeType.DocumentFragment, XmlNodeType.Notation);

implementation

uses System.Xml.Xsl, XmlConst;

{ Utility Functions }

function MakeNode(Node: XmlNode): IDOMNode;
const
  NodeClasses: array[System.Xml.XmlNodeType] of TCLRDOMNodeClass =
    (nil, TCLRDOMElement, TCLRDOMAttr, TCLRDOMText, TCLRDOMCDataSection,
     TCLRDOMEntityReference, TCLRDOMEntity, TCLRDOMProcessingInstruction,
     TCLRDOMComment, TCLRDOMDocument, TCLRDOMDocumentType, TCLRDOMDocumentFragment,
     TCLRDOMNotation, TCLRDOMText, TCLRDOMText, TCLRDOMElement, TCLRDOMEntity,
     TCLRDOMProcessingInstruction);
begin
  if Node <> nil then
    Result := NodeClasses[Node.nodeType].Create(Node) else
    Result := nil;
end;

function MakeNodeList(const NodeList: XmlNodeList): IDOMNodeList;
begin
  Result := TCLRDOMNodeList.Create(NodeList);
end;

function MakeNamedNodeMap(const NamedNodeMap: XmlNamedNodeMap): IDOMNamedNodeMap;
begin
  Result := TCLRDOMNamedNodeMap.Create(NamedNodeMap);
end;

function GetCLRNode(const Node: IDOMNode): XmlNode;
begin
  if not Assigned(Node) then
    raise DOMException.Create(SNodeExpected);
  Result := (Node as XmlNodeRef).GetXMLDOMNode;
end;

{ TCLRDOMImplementation }

constructor TCLRDOMImplementation.Create(DOMImpl: XmlImplementation);
begin
  inherited Create;
  FCLRDOMImpl := DOMImpl;
end;

function TCLRDOMImplementation.createDocument(const namespaceURI,
  qualifiedName: DOMString; doctype: IDOMDocumentType): IDOMDocument;
begin
  Result := TCLRDOMDocument.Create(XMLDocument.Create);
end;

function TCLRDOMImplementation.createDocumentType(const qualifiedName,
  publicId, systemId: DOMString): IDOMDocumentType;
begin
  DOMVendorNotSupported('createDocumentType', SCLRXML); { Do not localize }
end;

function TCLRDOMImplementation.hasFeature(const feature,
  version: DOMString): WordBool;
begin
  if not Assigned(FCLRDOMImpl) then
    FCLRDOMImpl := XmlImplementation.Create();
  Result := CLRDOMImpl.hasFeature(feature, version);
end;

{ TCLRDOMNode }

constructor TCLRDOMNode.Create(ANode: XmlNode);
begin
  Assert(Assigned(ANode));
  FCLRNode := ANode;
  inherited Create;
end;

function TCLRDOMNode.appendChild(const newChild: IDOMNode): IDOMNode;
var
  NewCLRChild,
  ReturnedChild: XmlNode;
begin
  NewCLRChild := GetCLRNode(newChild);
  ReturnedChild := CLRNode.appendChild(NewCLRChild);
  if ReturnedChild = NewCLRChild then
    Result := newChild else
    Result := MakeNode(ReturnedChild);
end;

function TCLRDOMNode.cloneNode(deep: WordBool): IDOMNode;
begin
  Result := MakeNode(CLRNode.cloneNode(deep));
end;

function TCLRDOMNode.get_attributes: IDOMNamedNodeMap;
begin
  if not Assigned(FAttributes) and Assigned(CLRNode.attributes) then
    FAttributes := MakeNamedNodeMap(CLRNode.attributes);
  Result := FAttributes;
end;

function TCLRDOMNode.get_childNodes: IDOMNodeList;
begin
  if not Assigned(FChildNodes) then
    FChildNodes := MakeNodeList(CLRNode.childNodes);
  Result := FChildNodes;
end;

function TCLRDOMNode.get_firstChild: IDOMNode;
begin
  Result := MakeNode(CLRNode.firstChild);
end;

function TCLRDOMNode.get_lastChild: IDOMNode;
begin
  Result := MakeNode(CLRNode.lastChild);
end;

function TCLRDOMNode.get_localName: DOMString;
begin
  Result := CLRNode.localName; 
end;

function TCLRDOMNode.get_namespaceURI: DOMString;
begin
  Result := CLRNode.namespaceURI;
end;

function TCLRDOMNode.get_nextSibling: IDOMNode;
begin
  Result := MakeNode(CLRNode.nextSibling);
end;

function TCLRDOMNode.get_nodeName: DOMString;
begin
  Result := CLRNode.Name;
end;

function TCLRDOMNode.get_nodeType: DOMNodeType;
begin
  Result := Ord(CLRNode.nodeType);
end;

function TCLRDOMNode.get_nodeValue: DOMString;
begin
  Result := CLRNode.Value;
end;

function TCLRDOMNode.get_ownerDocument: IDOMDocument;
begin
  if not Assigned(FOwnerDocument) then
    FOwnerDocument := TCLRDOMDocument.Create(CLRNode.ownerDocument);
  Result := FOwnerDocument;
end;

function TCLRDOMNode.get_parentNode: IDOMNode;
begin
  Result := MakeNode(CLRNode.parentNode);
end;

function TCLRDOMNode.get_prefix: DOMString;
begin
  Result := CLRNode.prefix;
end;

function TCLRDOMNode.get_previousSibling: IDOMNode;
begin
  Result := MakeNode(CLRNode.previousSibling);
end;

function TCLRDOMNode.hasChildNodes: WordBool;
begin
  Result := CLRNode.hasChildNodes;
end;

function TCLRDOMNode.insertBefore(const newChild,
  refChild: IDOMNode): IDOMNode;
begin
  Result := MakeNode(CLRNode.insertBefore(GetCLRNode(newChild), GetCLRNode(refChild)));
end;

procedure TCLRDOMNode.normalize;
begin
  CLRNode.Normalize;
end;

function TCLRDOMNode.removeChild(const childNode: IDOMNode): IDOMNode;
begin
  Result := MakeNode(CLRNode.removeChild(GetCLRNode(childNode)));
end;

function TCLRDOMNode.replaceChild(const newChild,
  oldChild: IDOMNode): IDOMNode;
begin
  Result := MakeNode(CLRNode.replaceChild(GetCLRNode(newChild), GetCLRNode(oldChild)));
end;

procedure TCLRDOMNode.set_nodeValue(value: DOMString);
begin
  CLRNode.Value := value;
end;

function TCLRDOMNode.supports(const feature, version: DOMString): WordBool;
begin
  Result := CLRNode.Supports(feature, version);
end;

function TCLRDOMNode.GetXMLDOMNode: XmlNode;
begin
  Result := CLRNode;
end;

function TCLRDOMNode.selectNode(const nodePath: WideString): IDOMNode;
var
  Node: XmlNode;
begin
  Node := CLRNode.selectSingleNode(nodePath);
  if Assigned(Node) then
    Result := MakeNode(Node) else
    Result := nil;
end;

function TCLRDOMNode.selectNodes(const nodePath: WideString): IDOMNodeList;
var
  NodeList: XmlNodeList;
begin
  NodeList := CLRNode.selectNodes(nodePath);
  if Assigned(NodeList) then
    Result := MakeNodeList(NodeList) else
    Result := nil;
end;

{ IDOMNodeEx Interface }

function TCLRDOMNode.get_text: DOMString;
begin
  Result := CLRNode.InnerText;
end;

procedure TCLRDOMNode.set_text(const Value: DOMString);
begin
  CLRNode.InnerText := Value;
end;

function TCLRDOMNode.get_xml: DOMString;
begin
  Result := CLRNode.OuterXml;
end;

procedure TCLRDOMNode.transformNode(const stylesheet: IDOMNode; var output: WideString);
var
  Transform: XslTransform;
  Writer: TextWriter;
begin
  Transform := XslTransform.Create;
  Transform.Load(GetCLRNode(stylesheet));
  Writer := StringWriter.Create;
  Transform.Transform(CLRNode, nil, Writer);
  output := Writer.ToString;
end;

procedure TCLRDOMNode.transformNode(const stylesheet: IDOMNode; const output: IDOMDocument);
var
  Transform: XslTransform;
  Stream: MemoryStream;
begin
  Transform := XslTransform.Create;
  Transform.Load(GetCLRNode(stylesheet));
  Stream := MemoryStream.Create;
  Transform.Transform(CLRNode, nil, Stream);
  Stream.Position := 0;
  (output as IDOMPersist).LoadFromStream(Stream);
end;

{ TCLRDOMNodeList }

constructor TCLRDOMNodeList.Create(ANodeList: XmlNodeList);
begin
  inherited Create;
  FCLRNodeList := ANodeList;
end;

function TCLRDOMNodeList.get_item(index: Integer): IDOMNode;
begin
  Result := MakeNode(CLRNodeList.Item(index));
end;

function TCLRDOMNodeList.get_length: Integer;
begin
  Result := CLRNodeList.Count;
end;

{ TCLRDOMNamedNodeMap }

constructor TCLRDOMNamedNodeMap.Create(ANamedNodeMap: XmlNamedNodeMap);
begin
  inherited Create;
  FCLRNamedNodeMap := ANamedNodeMap;
end;

function TCLRDOMNamedNodeMap.get_item(index: Integer): IDOMNode;
begin
  Result := MakeNode(CLRNamedNodeMap.Item(index));
end;

function TCLRDOMNamedNodeMap.get_length: Integer;
begin
  Result := CLRNamedNodeMap.Count;
end;

function TCLRDOMNamedNodeMap.getNamedItem(const name: DOMString): IDOMNode;
begin
  Result := MakeNode(CLRNamedNodeMap.getNamedItem(name));
end;

function TCLRDOMNamedNodeMap.getNamedItemNS(const namespaceURI,
  localName: DOMString): IDOMNode;
begin
  Result := MakeNode(CLRNamedNodeMap.getNamedItem(localName, namespaceURI));
end;

function TCLRDOMNamedNodeMap.removeNamedItem(const name: DOMString): IDOMNode;
begin
  Result := MakeNode(CLRNamedNodeMap.removeNamedItem(name));
end;

function TCLRDOMNamedNodeMap.removeNamedItemNS(const namespaceURI,
  localName: DOMString): IDOMNode;
begin
  Result := MakeNode(CLRNamedNodeMap.removeNamedItem(localName, namespaceURI));
end;

function TCLRDOMNamedNodeMap.setNamedItem(const newItem: IDOMNode): IDOMNode;
begin
  Result := MakeNode(CLRNamedNodeMap.setNamedItem(GetCLRNode(newItem)));
end;

function TCLRDOMNamedNodeMap.setNamedItemNS(const arg: IDOMNode): IDOMNode;
begin
  Result := MakeNode(CLRNamedNodeMap.setNamedItem(GetCLRNode(arg)));
end;

{ TCLRDOMCharacterData }

function TCLRDOMCharacterData.GetCLRCharacterData: XmlCharacterData;
begin
  Result := CLRNode as XmlCharacterData;
end;

procedure TCLRDOMCharacterData.appendData(const data: DOMString);
begin
  CLRCharacterData.appendData(data);
end;

procedure TCLRDOMCharacterData.deleteData(offset, count: Integer);
begin
  CLRCharacterData.deleteData(offset, count);
end;

function TCLRDOMCharacterData.get_data: DOMString;
begin
  Result := CLRCharacterData.data;
end;

function TCLRDOMCharacterData.get_length: Integer;
begin
  Result := CLRCharacterData.length;
end;

procedure TCLRDOMCharacterData.insertData(offset: Integer;
  const data: DOMString);
begin
  CLRCharacterData.insertData(offset, data);
end;

procedure TCLRDOMCharacterData.replaceData(offset, count: Integer;
  const data: DOMString);
begin
  CLRCharacterData.replaceData(offset, count, data);
end;

procedure TCLRDOMCharacterData.set_data(const data: DOMString);
begin
  CLRCharacterData.data := data;
end;

function TCLRDOMCharacterData.substringData(offset,
  count: Integer): DOMString;
begin
  Result := CLRCharacterData.Substring(offset, count);
end;

{ TCLRDOMAttr }

function TCLRDOMAttr.GetCLRAttribute: XmlAttribute;
begin
  Result := CLRNode as XmlAttribute;
end;

function TCLRDOMAttr.get_name: DOMString;
begin
  Result := CLRAttribute.name;
end;

function TCLRDOMAttr.get_ownerElement: IDOMElement;
begin
  Result := MakeNode(CLRAttribute.OwnerElement) as IDOMElement;
end;

function TCLRDOMAttr.get_specified: WordBool;
begin
  Result := CLRAttribute.specified;
end;

function TCLRDOMAttr.get_value: DOMString;
begin
  Result := CLRAttribute.value;
end;

procedure TCLRDOMAttr.set_value(const attributeValue: DOMString);
begin
  CLRAttribute.value := attributeValue;
end;

{ TCLRDOMElement }

function TCLRDOMElement.GetCLRElement: XmlElement;
begin
  Result := CLRNode as XmlElement;
end;

function TCLRDOMElement.get_tagName: DOMString;
begin
  Result := CLRElement.Name;
end;

function TCLRDOMElement.getAttribute(const name: DOMString): DOMString;
begin
  Result := VarToWideStr(CLRElement.getAttribute(name));
end;

function TCLRDOMElement.getAttributeNS(const namespaceURI,
  localName: DOMString): DOMString;
var
  AttrNode: IDOMAttr;
begin
  AttrNode := getAttributeNodeNS(namespaceURI, localName);
  if Assigned(AttrNode) then
    Result := VarToWideStr(AttrNode.NodeValue)
  else
    Result := '';
end;

function TCLRDOMElement.getAttributeNode(const name: DOMString): IDOMAttr;
begin
  Result := MakeNode(CLRElement.getAttributeNode(name)) as IDOMAttr;
end;

function TCLRDOMElement.getAttributeNodeNS(const namespaceURI,
  localName: DOMString): IDOMAttr;
begin
  Result := MakeNode(CLRElement.Attributes.getNamedItem(localName, namespaceURI)) as IDOMAttr;
end;

function TCLRDOMElement.getElementsByTagName(const name: DOMString): IDOMNodeList;
begin
  Result := MakeNodeList(CLRElement.getElementsByTagName(name));
end;

function TCLRDOMElement.getElementsByTagNameNS(const namespaceURI,
  localName: DOMString): IDOMNodeList;
begin
  Result := MakeNodeList(CLRElement.getElementsByTagName(localname, namespaceURI));
end;

function TCLRDOMElement.hasAttribute(const name: DOMString): WordBool;
begin
  Result := CLRElement.hasAttribute(name);
end;

function TCLRDOMElement.hasAttributeNS(const namespaceURI,
  localName: DOMString): WordBool;
begin
  Result := CLRElement.hasAttribute(localname, namespaceURI);
end;

procedure TCLRDOMElement.removeAttribute(const name: DOMString);
begin
  CLRElement.removeAttribute(name);
end;

function TCLRDOMElement.removeAttributeNode(const oldAttr: IDOMAttr): IDOMAttr;
begin
  Result := MakeNode(CLRElement.removeAttributeNode(
    GetCLRNode(oldAttr) as XmlAttribute)) as IDOMAttr;
end;

procedure TCLRDOMElement.removeAttributeNS(const namespaceURI,
  localName: DOMString);
begin
  CLRElement.removeAttribute(localName, namespaceURI);
end;

procedure TCLRDOMElement.setAttribute(const name, value: DOMString);
begin
  CLRElement.setAttribute(name, value);
end;

function TCLRDOMElement.setAttributeNode(const newAttr: IDOMAttr): IDOMAttr;
begin
  Result := MakeNode(CLRElement.setAttributeNode(
    GetCLRNode(newAttr) as XmlAttribute)) as IDOMAttr;
end;

function TCLRDOMElement.setAttributeNodeNS(const newAttr: IDOMAttr): IDOMAttr;
begin
  Result := setAttributeNode(newAttr);
end;

procedure TCLRDOMElement.setAttributeNS(const namespaceURI, qualifiedName, value: DOMString);
var
  Prefix: DOMString;
  AttrNode: XmlAttribute;
begin
  Prefix := ExtractPrefix(qualifiedName);
  if Prefix = '' then
    CLRElement.SetAttribute(ExtractLocalName(qualifiedName), namespaceURI, value)
  else
  begin
    AttrNode := CLRNode.ownerDocument.CreateNode(XmlNodeType.Attribute, Prefix,
                 ExtractLocalName(qualifiedName), namespaceURI) as XmlAttribute;
    AttrNode.Value := value;
    CLRElement.setAttributeNode(AttrNode);
  end;
end;

procedure TCLRDOMElement.normalize;
begin
  CLRElement.normalize;
end;

{ TCLRDOMText }

function TCLRDOMText.splitText(offset: Integer): IDOMText;
begin
  Result := MakeNode((CLRNode as XmlText).splitText(offset)) as IDOMText;
end;

{ TCLRDOMDocumentType }

function TCLRDOMDocumentType.GetCLRDocumentType: XmlDocumentType;
begin
  Result := CLRNode as XmlDocumentType;
end;

function TCLRDOMDocumentType.get_entities: IDOMNamedNodeMap;
begin
  if not Assigned(FEntities) then
    FEntities := MakeNamedNodeMap(CLRDocumentType.get_entities);
  Result := FEntities;
end;

function TCLRDOMDocumentType.get_internalSubset: DOMString;
begin
  Result := CLRDocumentType.InternalSubset;
end;

function TCLRDOMDocumentType.get_name: DOMString;
begin
  Result := CLRDocumentType.name;
end;

function TCLRDOMDocumentType.get_notations: IDOMNamedNodeMap;
begin
  if not Assigned(FNotations) then
    FNotations := MakeNamedNodeMap(CLRDocumentType.notations);
  Result := FNotations;
end;

function TCLRDOMDocumentType.get_publicId: DOMString;
begin
  Result := CLRDocumentType.PublicId;
end;

function TCLRDOMDocumentType.get_systemId: DOMString;
begin
  Result := CLRDocumentType.SystemId;
end;

{ TCLRDOMNotation }

function TCLRDOMNotation.GetCLRNotation: XmlNotation;
begin
  Result := CLRNode as XmlNotation;
end;

function TCLRDOMNotation.get_publicId: DOMString;
begin
  Result := CLRNotation.publicId;
end;

function TCLRDOMNotation.get_systemId: DOMString;
begin
  Result := CLRNotation.systemId;
end;

{ TCLRDOMEntity }

function TCLRDOMEntity.GetCLREntity: XmlEntity;
begin
  Result := CLRNode as XmlEntity;
end;

function TCLRDOMEntity.get_notationName: DOMString;
begin
  Result := CLREntity.notationName;
end;

function TCLRDOMEntity.get_publicId: DOMString;
begin
  Result := CLREntity.publicId;
end;

function TCLRDOMEntity.get_systemId: DOMString;
begin
  Result := CLREntity.systemId;
end;

{ TCLRDOMProcessingInstruction }

function TCLRDOMProcessingInstruction.GetCLRProcessingInstruction: XmlProcessingInstruction;
begin
  Result := CLRNode as XmlProcessingInstruction;
end;

function TCLRDOMProcessingInstruction.get_data: DOMString;
begin
  Result := CLRProcessingInstruction.data;
end;

function TCLRDOMProcessingInstruction.get_target: DOMString;
begin
  Result := CLRProcessingInstruction.target;
end;

procedure TCLRDOMProcessingInstruction.set_data(const value: DOMString);
begin
  CLRProcessingInstruction.data := value;
end;

{ TCLRDOMDocument }

function TCLRDOMDocument.GetCLRDocument: XmlDocument;
begin
  Result := CLRNode as XmlDocument;
end;

function TCLRDOMDocument.createAttribute(const name: DOMString): IDOMAttr;
begin
  Result := MakeNode(CLRDocument.createAttribute(name)) as IDOMAttr;
end;

function TCLRDOMDocument.createAttributeNS(const namespaceURI,
  qualifiedName: DOMString): IDOMAttr;
begin
  Result := MakeNode(CLRDocument.createAttribute(qualifiedName, namespaceURI)) as IDOMAttr;
end;

function TCLRDOMDocument.createCDATASection(const data: DOMString): IDOMCDATASection;
begin
  Result := TCLRDOMCDATASection.Create(CLRDocument.createCDATASection(data)) as IDOMCDATASection;
end;

function TCLRDOMDocument.createComment(const data: DOMString): IDOMComment;
begin
  Result := TCLRDOMComment.Create(CLRDocument.createComment(data)) as IDOMComment;
end;

function TCLRDOMDocument.createDocumentFragment: IDOMDocumentFragment;
begin
  Result := TCLRDOMDocumentFragment.Create(CLRDocument.createDocumentFragment) as IDOMDocumentFragment;
end;

function TCLRDOMDocument.createElement(const tagName: DOMString): IDOMElement;
begin
  Result := MakeNode(CLRDocument.createElement(tagName)) as IDOMElement;
end;

function TCLRDOMDocument.createElementNS(const namespaceURI, qualifiedName: DOMString): IDOMElement;
begin
  Result := MakeNode(CLRDocument.createElement(qualifiedName, namespaceURI)) as IDOMElement;
end;

function TCLRDOMDocument.createEntityReference(const name: DOMString): IDOMEntityReference;
begin
  Result := IDOMEntityReference(TCLRDOMEntityReference.Create(CLRDocument.createEntityReference(name)));
end;

function TCLRDOMDocument.createProcessingInstruction(const target, data: DOMString): IDOMProcessingInstruction;
begin
  Result := IDOMProcessingInstruction(TCLRDOMProcessingInstruction.Create(CLRDocument.createProcessingInstruction(target, data)));
end;

function TCLRDOMDocument.createTextNode(const data: DOMString): IDOMText;
begin
  Result := MakeNode(CLRDocument.createTextNode(data)) as IDOMText;
end;

function TCLRDOMDocument.get_doctype: IDOMDocumentType;
begin
  Result := IDOMDocumentType(TCLRDOMDocumentType.Create(CLRDocument.DocumentType));
end;

function TCLRDOMDocument.get_documentElement: IDOMElement;
begin
  Result := MakeNode(CLRDocument.documentElement) as IDOMElement;
end;

function TCLRDOMDocument.get_domImplementation: IDOMImplementation;
begin
  Result := Xmldom.IDOMImplementation(TCLRDOMImplementation.Create(CLRDocument.get_Implementation));
end;

function TCLRDOMDocument.getElementById(const elementId: DOMString): IDOMElement;
begin
  Result := MakeNode(CLRDocument.GetElementById(elementId)) as IDOMElement;
end;

function TCLRDOMDocument.getElementsByTagName(const tagName: DOMString): IDOMNodeList;
begin
  Result := MakeNodeList(CLRDocument.getElementsByTagName(tagName));
end;

function TCLRDOMDocument.getElementsByTagNameNS(const namespaceURI, localName: DOMString): IDOMNodeList;
begin
  Result := MakeNodeList(CLRDocument.getElementsByTagName(localName, namespaceURI));
end;

function TCLRDOMDocument.importNode(importedNode: IDOMNode; deep: WordBool): IDOMNode;
begin
  Result := MakeNode(CLRDocument.importNode(GetCLRNode(importedNode), deep));
end;

procedure TCLRDOMDocument.set_documentElement(const docElement: IDOMElement);
begin
  if Assigned(CLRDocument.documentElement) then
    CLRDocument.removeChild(CLRDocument.documentElement);
  if Assigned(docElement) then
    CLRDocument.AppendChild(GetCLRNode(docElement) as XmlElement);
end;

{ IDOMParseOptions Interface }

function TCLRDOMDocument.get_async: Boolean;
begin
  Result := False;
end;

procedure TCLRDOMDocument.set_async(Value: Boolean);
begin
  if Value then
    DOMVendorNotSupported('set_async', SCLRXML); { Do not localize }
end;

function TCLRDOMDocument.get_preserveWhiteSpace: Boolean;
begin
  Result := CLRDocument.Get_preserveWhiteSpace;
end;

function TCLRDOMDocument.get_resolveExternals: Boolean;
begin
  Result := FResolveExternals;

end;

function TCLRDOMDocument.get_validate: Boolean;
begin
  Result := FValidate;
end;

procedure TCLRDOMDocument.set_preserveWhiteSpace(Value: Boolean);
begin
  CLRDocument.Set_preserveWhiteSpace(Value);
end;

procedure TCLRDOMDocument.set_resolveExternals(Value: Boolean);
begin
  FResolveExternals := Value;
end;

procedure TCLRDOMDocument.set_validate(Value: Boolean);
begin
  FValidate := Value;
end;

{ IDOMPersist interface }

function TCLRDOMDocument.asyncLoadState: Integer;
begin
  Result := 0;
end;

function TCLRDOMDocument.get_xml: DOMString;
var
  Writer: StringWriter;
begin
  // This is coded this way instead of using the OuterXml property to make sure the endcoding
  // value is set to utf-16 and not utf-8.
  Writer := StringWriter.Create;
  CLRDocument.Save(Writer);
  Result := Writer.ToString;
end;

function TCLRDOMDocument.InternalLoad(Reader: XmlReader): WordBool;
begin
  try
    if FValidate then
      Reader := XmlValidatingReader.Create(Reader);
    CLRDocument.load(Reader);
    Result := True;
  except
    on e: System.SystemException do
    begin
      FParseError := e;
      Result := False;
    end
    else
      raise;
  end;
end;

function TCLRDOMDocument.load(source: OleVariant): WordBool;
var
  Reader: XmlTextReader;
begin
  // We only support load from a string that is a file name.
  if VarType(source) = varString then
  begin
    Reader := XmlTextReader.Create(VarToStr(source));
    try
      Result := InternalLoad(Reader);
    finally
      Reader.Close;
    end;
  end
  else
    Result := False;
end;

function TCLRDOMDocument.loadFromStream(const stream: TStream): WordBool;
begin
  Result := loadFromStream(TStreamToCLRStream.GetStream(stream));
end;

procedure TCLRDOMDocument.saveToStream(const stream: TStream);
begin
  CLRDocument.save(TStreamToCLRStream.GetStream(stream));
end;

procedure TCLRDOMDocument.save(destination: OleVariant);
begin
  if VarType(destination) = varString then
    CLRDocument.save(VarToStr(destination));
end;

function TCLRDOMDocument.loadxml(const Value: DOMString): WordBool;
var
  Reader: XmlTextReader;
begin
  Reader := XmlTextReader.Create(StringReader.Create(Value));
  try
    Result := InternalLoad(Reader);
  finally
    Reader.Close;
  end;
end;

procedure TCLRDOMDocument.set_OnAsyncLoad(const Sender: TObject;
  EventHandler: TAsyncEventHandler);
begin
  DOMVendorNotSupported('set_OnAsyncLoad', SCLRXML); { Do not localize }
end;

function TCLRDOMDocument.loadFromStream(const stream: System.IO.Stream): WordBool;
var
  Reader: XmlTextReader;
begin
  Reader := XmlTextReader.Create(stream);
  try
    Result := InternalLoad(Reader);
  finally
    Reader.Close;
  end;
end;

procedure TCLRDOMDocument.saveToStream(const stream: System.IO.Stream);
begin
  CLRDocument.save(stream);
end;

{ IDOMParseError }

function TCLRDOMDocument.get_errorCode: Integer;
begin
  { Not implemented in System.Xml.XmlException }
  Result := -1;
end;

function TCLRDOMDocument.get_filepos: Integer;
begin
  { Not implemented in System.Xml.XmlException }
  Result := -1;
end;

function TCLRDOMDocument.get_line: Integer;
begin
  if (FParseError is XmlException) then
    Result := (FParseError as XmlException).LineNumber
  else if (FParseError is XmlSchemaException) then
    Result := (FParseError as XmlSchemaException).LineNumber
  else
    Result := -1;
end;

function TCLRDOMDocument.get_linepos: Integer;
begin
  if (FParseError is XmlException) then
    Result := (FParseError as XmlException).LinePosition
  else if (FParseError is XmlSchemaException) then
    Result := (FParseError as XmlSchemaException).LinePosition
  else
    Result := -1;
end;

function TCLRDOMDocument.get_reason: WideString;
begin
  if (FParseError is XmlException) then
    Result := (FParseError as XmlException).Message
  else if (FParseError is XmlSchemaException) then
    Result := (FParseError as XmlSchemaException).Message
end;

function TCLRDOMDocument.get_srcText: WideString;
begin
  { Not implemented in System.Xml.XmlException }
  Result := '';
end;

function TCLRDOMDocument.get_url: WideString;
begin
  if (FParseError is XmlSchemaException) then
    Result := (FParseError as XmlSchemaException).SourceUri;
end;

{ IDOMXMLProlog }

function TCLRDOMDocument.get_Encoding: DOMString;
begin
  if CLRDocument.FirstChild is XmlDeclaration then
    Result := (CLRDocument.FirstChild as XmlDeclaration).Encoding;
end;

function TCLRDOMDocument.get_Standalone: DOMString;
begin
  if CLRDocument.FirstChild is XmlDeclaration then
    Result := (CLRDocument.FirstChild as XmlDeclaration).StandAlone;
end;

function TCLRDOMDocument.get_Version: DOMString;
begin
  if CLRDocument.FirstChild is XmlDeclaration then
    Result := (CLRDocument.FirstChild as XmlDeclaration).Version;
end;

procedure TCLRDOMDocument.set_Encoding(const Value: DOMString);
begin
  if CLRDocument.FirstChild is XmlDeclaration then
    (CLRDocument.FirstChild as XmlDeclaration).Encoding := Value
  else if Value <> '' then
    CLRDocument.InsertBefore(CLRDocument.CreateXMLDeclaration('1.0', Value, ''),
      CLRDocument.FirstChild);
end;

procedure TCLRDOMDocument.set_Standalone(const Value: DOMString);
begin
  if CLRDocument.FirstChild is XmlDeclaration then
    (CLRDocument.FirstChild as XmlDeclaration).Standalone := Value
  else if Value <> '' then
    CLRDocument.InsertBefore(CLRDocument.CreateXMLDeclaration('1.0','', Value),
      CLRDocument.FirstChild);
end;

procedure TCLRDOMDocument.set_Version(const Value: DOMString);
begin
  if CLRDocument.FirstChild is XmlDeclaration then
  begin
    { Blanking the Version when no standalone or encoding removes the XmlDeclaration }
    with (CLRDocument.FirstChild as XmlDeclaration) do
    if (Value = '') and (StandAlone = '') and (Encoding = '') then
      CLRDocument.RemoveChild(CLRDocument.FirstChild);
    { Otherwise do nothing since it already is set and is readonly }
  end
  else if Value <> '' then
    { Create a new XmlDeclaration using the value passed in which should be 1.0 }
    CLRDocument.InsertBefore(CLRDocument.CreateXMLDeclaration(Value, '', ''),
      CLRDocument.FirstChild);
end;

{ TCLRDOMImplementationFactory }

function TCLRDOMImplementationFactory.DOMImplementation: IDOMImplementation;
begin
  Result := TCLRDOMImplementation.Create(nil);
end;

function TCLRDOMImplementationFactory.Description: String;
begin
  Result := SCLRXML;
end;

initialization
  CLRXML_DOM := TCLRDOMImplementationFactory.Create;
  RegisterDOMVendor(CLRXML_DOM);
finalization
  UnRegisterDOMVendor(CLRXML_DOM);
  CLRXML_DOM.Free;
end.
